<?php
/**
 * دوال الإحصائيات
 */

/**
 * تتبع الزيارات
 */
function trackVisit() {
    global $conn;
    
    try {
        // تعيين الترميز للاتصال
        $conn->set_charset("utf8mb4");
        
        // جمع معلومات الزيارة
        $ip = $_SERVER['REMOTE_ADDR'];
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $pageUrl = $_SERVER['REQUEST_URI'] ?? '';
        $sessionId = session_id();
        
        // الحصول على معلومات البلد والمدينة
        $country = 'غير معروف';
        $city = 'غير معروف';
        
        try {
            $ipInfo = @file_get_contents("http://ip-api.com/json/{$ip}?fields=country,city&lang=ar");
            if ($ipInfo) {
                $ipData = json_decode($ipInfo);
                if ($ipData && isset($ipData->country)) {
                    $country = $ipData->country;
                    $city = $ipData->city ?? 'غير معروف';
                }
            }
        } catch (Exception $e) {
            error_log("IP API Error: " . $e->getMessage());
        }

        // تسجيل الزيارة
        $stmt = $conn->prepare("
            INSERT INTO visitors 
            (ip_address, user_agent, page_url, country, city, session_id) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }
        
        $stmt->bind_param("ssssss", $ip, $userAgent, $pageUrl, $country, $city, $sessionId);
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        $stmt->close();
        
        return true;
        
    } catch (Exception $e) {
        error_log("Error tracking visit: " . $e->getMessage());
        return false;
    }
}

/**
 * تتبع النشاطات
 */
function trackActivity($type, $contentId = null, $details = '') {
    global $conn;
    
    try {
        // تعيين الترميز للاتصال
        $conn->set_charset("utf8mb4");
        
        // التحقق من نوع النشاط
        $validTypes = ['copy', 'search', 'report'];
        if (!in_array($type, $validTypes)) {
            return false;
        }
        
        // جمع المعلومات
        $ip = $_SERVER['REMOTE_ADDR'];
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // الحصول على معلومات البلد
        $country = 'غير معروف';
        try {
            $ipInfo = @file_get_contents("http://ip-api.com/json/{$ip}?fields=country&lang=ar");
            if ($ipInfo) {
                $ipData = json_decode($ipInfo);
                if ($ipData && isset($ipData->country)) {
                    $country = $ipData->country;
                }
            }
        } catch (Exception $e) {
            error_log("IP API Error in trackActivity: " . $e->getMessage());
        }

        // تسجيل النشاط
        $stmt = $conn->prepare("
            INSERT INTO activities 
            (activity_type, content_id, ip_address, user_agent, country, details) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }
        
        $stmt->bind_param("sissss", $type, $contentId, $ip, $userAgent, $country, $details);
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        $stmt->close();
        
        return true;
        
    } catch (Exception $e) {
        error_log("Error tracking activity: " . $e->getMessage());
        return false;
    }
}

/**
 * جلب إحصائيات الزيارات
 */
function getVisitStats($dateCondition) {
    global $conn;
    
    $stats = [
        'total' => 0,
        'unique' => 0,
        'returning' => 0,
        'average_daily' => 0
    ];
    
    // إجمالي الزيارات
    $result = $conn->query("SELECT COUNT(*) as count FROM visitors WHERE {$dateCondition}");
    if ($result) {
        $stats['total'] = $result->fetch_assoc()['count'];
    }
    
    // الزوار الفريدين
    $result = $conn->query("SELECT COUNT(DISTINCT ip_address) as count FROM visitors WHERE {$dateCondition}");
    if ($result) {
        $stats['unique'] = $result->fetch_assoc()['count'];
    }
    
    // الزوار العائدين
    $result = $conn->query("
        SELECT COUNT(*) as count 
        FROM (
            SELECT ip_address 
            FROM visitors 
            WHERE {$dateCondition} 
            GROUP BY ip_address 
            HAVING COUNT(*) > 1
        ) as rv
    ");
    if ($result) {
        $stats['returning'] = $result->fetch_assoc()['count'];
    }
    
    // متوسط الزيارات اليومية
    $result = $conn->query("
        SELECT ROUND(COUNT(*) / COUNT(DISTINCT DATE(visit_date)), 0) as avg 
        FROM visitors 
        WHERE {$dateCondition}
    ");
    if ($result) {
        $stats['average_daily'] = $result->fetch_assoc()['avg'];
    }
    
    return $stats;
}

/**
 * جلب إحصائيات النشاطات
 */
function getActivityStats($dateCondition) {
    global $conn;
    
    return [
        'copies' => $conn->query("
            SELECT COUNT(*) as count 
            FROM activities 
            WHERE activity_type = 'copy' 
            AND {$dateCondition}
        ")->fetch_assoc()['count'] ?? 0,
        
        'searches' => $conn->query("
            SELECT COUNT(*) as count 
            FROM activities 
            WHERE activity_type = 'search' 
            AND {$dateCondition}
        ")->fetch_assoc()['count'] ?? 0,
        
        'reports' => $conn->query("
            SELECT COUNT(*) as count 
            FROM activities 
            WHERE activity_type = 'report' 
            AND {$dateCondition}
        ")->fetch_assoc()['count'] ?? 0
    ];
}

/**
 * جلب توزيع البلدان
 */
function getCountryStats($dateCondition) {
    global $conn;
    
    return $conn->query("
        SELECT 
            country,
            COUNT(*) as count 
        FROM visitors 
        WHERE {$dateCondition}
        GROUP BY country 
        ORDER BY count DESC 
        LIMIT 10
    ")->fetch_all(MYSQLI_ASSOC);
}

/**
 * جلب توزيع المتصفحات
 */
function getBrowserStats($dateCondition) {
    global $conn;
    
    return $conn->query("
        SELECT 
            CASE 
                WHEN user_agent LIKE '%Chrome%' THEN 'Chrome'
                WHEN user_agent LIKE '%Firefox%' THEN 'Firefox'
                WHEN user_agent LIKE '%Safari%' THEN 'Safari'
                WHEN user_agent LIKE '%Edge%' THEN 'Edge'
                ELSE 'Other'
            END as browser,
            COUNT(*) as count
        FROM visitors 
        WHERE {$dateCondition}
        GROUP BY browser
        ORDER BY count DESC
    ")->fetch_all(MYSQLI_ASSOC);
}

/**
 * جلب إحصائيات المحتوى
 */
function getContentStats($dateCondition) {
    global $conn;
    
    return [
        'most_copied' => $conn->query("
            SELECT 
                c.title,
                COUNT(a.id) as count 
            FROM activities a 
            JOIN contents c ON a.content_id = c.id 
            WHERE a.activity_type = 'copy' 
            AND {$dateCondition}
            GROUP BY c.id, c.title 
            ORDER BY count DESC 
            LIMIT 5
        ")->fetch_all(MYSQLI_ASSOC),
        
        'most_searched' => $conn->query("
            SELECT 
                c.title,
                COUNT(a.id) as count 
            FROM activities a 
            JOIN contents c ON a.content_id = c.id 
            WHERE a.activity_type = 'search' 
            AND {$dateCondition}
            GROUP BY c.id, c.title 
            ORDER BY count DESC 
            LIMIT 5
        ")->fetch_all(MYSQLI_ASSOC)
    ];
}

/**
 * جلب إحصائيات الوقت
 */
function getTimeStats($dateCondition) {
    global $conn;
    
    return [
        'hourly' => $conn->query("
            SELECT 
                HOUR(visit_date) as hour,
                COUNT(*) as count 
            FROM visitors 
            WHERE {$dateCondition}
            GROUP BY HOUR(visit_date)
            ORDER BY hour
        ")->fetch_all(MYSQLI_ASSOC),
        
        'daily' => $conn->query("
            SELECT 
                DATE(visit_date) as date,
                COUNT(*) as total,
                COUNT(DISTINCT ip_address) as unique_visitors
            FROM visitors 
            WHERE visit_date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
            GROUP BY DATE(visit_date)
            ORDER BY date
        ")->fetch_all(MYSQLI_ASSOC)
    ];
}
?>